//! HTTP API of Obnam server.

use std::sync::{Arc, Mutex};

use axum::{
    Json, Router, debug_handler,
    extract::{FromRequestParts, State},
    http::{StatusCode, header::AUTHORIZATION},
    routing::get,
};

use crate::{
    config::ServerConfig,
    token::{ObnamTrustedToken, TokenFactory},
};

pub struct HttpApi {
    listen: String,
    api_state: ApiState,
}

impl HttpApi {
    pub async fn serve(&self) -> Result<(), ApiError> {
        let app = Router::new()
            .route("/validate", get(validate))
            .with_state(self.api_state.clone());
        let listener = tokio::net::TcpListener::bind(&self.listen)
            .await
            .map_err(|err| ApiError::Bind(self.listen.clone(), err))?;
        axum::serve(listener, app).await.map_err(ApiError::Serve)?;
        Ok(())
    }
}

impl TryFrom<&ServerConfig> for HttpApi {
    type Error = ApiError;
    fn try_from(config: &ServerConfig) -> Result<Self, Self::Error> {
        Ok(Self {
            listen: config.listen().to_string(),
            api_state: ApiState::try_from(config)?,
        })
    }
}

#[debug_handler]
async fn validate(
    _: State<ApiState>,
    TokenExtractor(token): TokenExtractor,
) -> Json<ObnamTrustedToken> {
    Json(token)
}

struct TokenExtractor(ObnamTrustedToken);

impl FromRequestParts<ApiState> for TokenExtractor {
    type Rejection = (StatusCode, &'static str);
    async fn from_request_parts(
        parts: &mut axum::http::request::Parts,
        state: &ApiState,
    ) -> Result<Self, Self::Rejection> {
        if let Some(authz) = parts.headers.get(AUTHORIZATION)
            && let Ok(authz) = String::from_utf8(authz.as_bytes().to_vec())
            && let Some(untrusted) = authz.strip_prefix("Bearer ")
        {
            let state = state.state.lock().expect("lock API state");
            if let Ok(trusted) = state.token_factory.validate(untrusted) {
                Ok(TokenExtractor(trusted))
            } else {
                Err((StatusCode::UNAUTHORIZED, "Unauthorized"))
            }
        } else {
            Err((StatusCode::BAD_REQUEST, "`Authorization` header is missing"))
        }
    }
}

struct UnlockedState {
    token_factory: TokenFactory,
}

#[derive(Clone)]
struct ApiState {
    state: Arc<Mutex<UnlockedState>>,
}

impl TryFrom<&ServerConfig> for ApiState {
    type Error = ApiError;
    fn try_from(config: &ServerConfig) -> Result<Self, Self::Error> {
        Ok(Self {
            state: Arc::new(Mutex::new(UnlockedState {
                token_factory: TokenFactory::try_from(config).map_err(ApiError::TokenFactory)?,
            })),
        })
    }
}

#[derive(Debug, thiserror::Error)]
pub enum ApiError {
    #[error("failed to bind to {0:?}")]
    Bind(String, #[source] std::io::Error),

    #[error("failed to serve HTTP")]
    Serve(#[source] std::io::Error),

    #[error("failed to create a token factory from configuration")]
    TokenFactory(#[source] crate::token::TokenFactoryError),
}
