//! Fundamental components for implementing backup software.
//!
//! This is the third generation of Obnam. The first two generations
//! were backup software. This generation is an exploration of how the
//! fundamental components of backup software can be implemented well.
//! This is not a backup program, but it may some day grow into one.
//! For now, it's a fun hobby project for me to think about how to
//! implement backup software.
//!
//! # Example
//!
//! ```rust
//! # use obnam::{
//! #     cipher::*,
//! #     chunk::*,
//! #     credential::*,
//! #     plaintext::*,
//! # };
//! # fn main() -> Result<(), Box<dyn std::error::Error>> {
//! // Create an encrypted data chunk and serialize it as if for persistent storage.
//! let engine = Engine::new(EngineKind::Aead, Key::default());
//! let plaintext = Plaintext::uncompressed(b"hello, world");
//! let meta = Metadata::new(Id::default(), Label::from("data-chunk"));
//! let chunk = Chunk::data(DataChunk::encrypt(&engine, &plaintext, meta.clone())?);
//! let ser = chunk.serialize()?;
//!
//! // De-serialize data chunk as if it had been read from persistent storage.
//! let chunk = Chunk::parse(&ser)?;
//! let data = match chunk {
//!    Chunk::Data(data) => data,
//!    _ => panic!(),
//! };
//!
//! // De-crypt data chunk.
//! let decrypted = data.decrypt(&engine)?;
//! assert_eq!(plaintext, decrypted);
//! # Ok(())
//! # }
//! ```

#![deny(missing_docs)]

pub mod chunk;
pub mod cipher;
pub mod client;
pub mod config;
pub mod credential;
pub mod plaintext;
pub mod sop;
pub mod store;
pub mod util;
