use std::collections::HashMap;

use clap::Parser;

use log::debug;
use obnam::{
    chunk::{Chunk, ChunkError, DataChunk, Id, Metadata},
    cipher::{Engine, EngineKind},
    client::ClientChunk,
    config::Config,
    credential::Credential,
    store::{ChunkStore, StoreError},
};

use super::{Args, Leaf, MainError};

/// Create a new client chunk.
#[derive(Parser)]
pub struct InitClient {
    /// Name of the new client.
    client_name: String,

    /// Create a credential chunk using this credential specification
    /// in the configuration file.
    #[clap(long)]
    credential: Option<String>,
}

impl Leaf for InitClient {
    type Error = ClientCmdError;

    fn run(&self, args: &Args, config: &Config) -> Result<(), Self::Error> {
        debug!("Create a client chunk for client {}", self.client_name);
        let store = Self::open_repository(config)?;
        let mut client = ClientChunk::new(&self.client_name);

        let client_key = if let Ok(client_key) = Self::client_key(args) {
            client_key
        } else {
            client
                .generate_key("default")
                .map_err(ClientCmdError::GenerateKey)?
        };

        let engine = Engine::new(EngineKind::Aead, client_key.clone());

        let clients = clients(&engine, &store)?;
        if clients
            .iter()
            .any(|client| client.name() == self.client_name)
        {
            return Err(ClientCmdError::AleadyExists(self.client_name.clone()));
        }

        let plaintext = client.to_plaintext().map_err(ClientCmdError::ToPlaintext)?;
        let metadata = Metadata::from_label("client");

        let data =
            DataChunk::encrypt(&engine, &plaintext, metadata).map_err(ClientCmdError::Encrypt)?;
        let chunk = Chunk::Client(data);

        let store = Self::open_repository(config)?;
        store.add_chunk(&chunk).map_err(ClientCmdError::AddChunk)?;

        if let Some(cred) = &self.credential
            && let Some(spec) = config.credentials().get(cred)
        {
            debug!("Create a credential for client {}", self.client_name);
            let method = spec.method();
            let credential =
                Credential::new(&method, &client_key).map_err(ClientCmdError::NewCredential)?;
            let metadata = Metadata::from_label("credential");
            let chunk = Chunk::credential(metadata, credential);
            store.add_chunk(&chunk).map_err(ClientCmdError::AddChunk)?;
        }

        Ok(())
    }
}

/// List all client chunks.
#[derive(Parser)]
pub struct ListClients {}

impl Leaf for ListClients {
    type Error = ClientCmdError;

    fn run(&self, args: &Args, config: &Config) -> Result<(), Self::Error> {
        debug!("List all clients whose client chunks we can open");
        let store = Self::open_repository(config)?;
        let engine = Engine::new(
            EngineKind::Aead,
            Self::client_key_from_store(args, config, &store)?.clone(),
        );
        let clients = clients(&engine, &store)?;
        for client in clients {
            println!("{}", client.name());
        }
        Ok(())
    }
}

/// Show contents of a given client chunk.
#[derive(Parser)]
pub struct ShowClient {
    /// Name of client to show.
    client_name: String,
}

impl Leaf for ShowClient {
    type Error = ClientCmdError;

    fn run(&self, args: &Args, config: &Config) -> Result<(), Self::Error> {
        debug!("Show contents of client chunk for {}", self.client_name);
        let store = Self::open_repository(config)?;
        let engine = Engine::new(
            EngineKind::Aead,
            Self::client_key_from_store(args, config, &store)?.clone(),
        );
        let client = client(&engine, &store, &self.client_name)?;
        println!(
            "{}",
            serde_json::to_string_pretty(&client).map_err(ClientCmdError::ToJson)?
        );
        Ok(())
    }
}

/// Generate a new chunk key and add it to the client chunk.
#[derive(Parser)]
pub struct GenerateKey {
    /// Name of client.
    client_name: String,

    /// Name of new key.
    key_name: String,
}

impl Leaf for GenerateKey {
    type Error = ClientCmdError;

    fn run(&self, args: &Args, config: &Config) -> Result<(), Self::Error> {
        debug!(
            "Generate a new chunk key {} for client {}",
            self.key_name, self.client_name
        );
        let store = Self::open_repository(config)?;
        let engine = Engine::new(
            EngineKind::Aead,
            Self::client_key_from_store(args, config, &store)?.clone(),
        );
        let (id, existing) = client(&engine, &store, &self.client_name)?;

        let mut new = ClientChunk::new(existing.name());
        new.add_old_versions(existing.old_versions());
        new.add_old_versions(&[id]);
        new.generate_key(&self.key_name)
            .map_err(|err| ClientCmdError::Generate(self.client_name.clone(), err))?;
        assert_eq!(new.old_versions().len(), existing.old_versions().len() + 1);

        let plaintext = new.to_plaintext().map_err(ClientCmdError::ToPlaintext)?;
        let metadata = Metadata::from_label("client");
        let data =
            DataChunk::encrypt(&engine, &plaintext, metadata).map_err(ClientCmdError::Encrypt)?;
        let chunk = Chunk::Client(data);
        store.add_chunk(&chunk).map_err(ClientCmdError::AddChunk)?;

        Ok(())
    }
}

fn clients(engine: &Engine, store: &ChunkStore) -> Result<Vec<ClientChunk>, ClientCmdError> {
    let clients = store
        .find_client_chunks()
        .map_err(ClientCmdError::FindClient)?;

    let our_clients: Vec<ClientChunk> = clients
        .iter()
        .filter_map(|meta| store.get_client_chunk(meta.id()).ok())
        .filter_map(|chunk| match chunk {
            Chunk::Client(data) => Some(data),
            _ => None,
        })
        .filter_map(|chunk| chunk.decrypt(engine).ok())
        .filter_map(|plaintext| ClientChunk::try_from(&plaintext).ok())
        .collect();

    Ok(our_clients)
}

fn client(
    engine: &Engine,
    store: &ChunkStore,
    wanted: &str,
) -> Result<(Id, ClientChunk), ClientCmdError> {
    fn is_old(id: &Id, map: &HashMap<Id, ClientChunk>) -> bool {
        for client in map.values() {
            if client.old_versions().contains(id) {
                return true;
            }
        }
        false
    }

    let metas = store
        .find_client_chunks()
        .map_err(ClientCmdError::FindClient)?;

    let mut map: HashMap<Id, ClientChunk> = HashMap::new();
    for x in metas {
        let chunk = store.get_client_chunk(x.id()).unwrap();
        let plaintext = match chunk {
            Chunk::Client(data) => data.decrypt(engine).unwrap(),
            _ => return Err(ClientCmdError::NotClient),
        };
        let client = ClientChunk::try_from(&plaintext).unwrap();
        map.insert(x.id().clone(), client);
    }

    let mut roots: Vec<&Id> = map.keys().collect();
    loop {
        if roots.is_empty() {
            return Err(ClientCmdError::NoSuchClient(wanted.to_string()));
        }

        let mut found = None;
        for (i, id) in roots.iter().enumerate() {
            if is_old(id, &map) {
                found = Some(i);
                break;
            }
        }

        if let Some(i) = found {
            roots.remove(i);
            if roots.len() == 1 {
                break;
            }
        } else {
            break;
        }
    }

    match roots.len() {
        0 => Err(ClientCmdError::NoSuchClient(wanted.to_string())),
        1 => {
            let id = roots
                .first()
                .ok_or(ClientCmdError::NoSuchClient(wanted.to_string()))?;
            let c = map
                .get(id)
                .ok_or(ClientCmdError::NoSuchClient(wanted.to_string()))?;
            Ok(((*id).clone(), c.clone()))
        }
        _ => Err(ClientCmdError::TooManyClients(wanted.to_string())),
    }
}

#[derive(Debug, thiserror::Error)]
pub enum ClientCmdError {
    #[error(transparent)]
    Main(#[from] MainError),

    #[error("failed to add chunk to repository")]
    AddChunk(#[source] StoreError),

    #[error("failed to convert client chunk to plain text data")]
    ToPlaintext(#[source] obnam::client::ClientChunkError),

    #[error("failed to encrypt chunk")]
    Encrypt(#[source] ChunkError),

    #[error("failed to find client chunks")]
    FindClient(#[source] StoreError),

    #[error("can't create another client named {0}")]
    AleadyExists(String),

    #[error("no client found named {0}")]
    NoSuchClient(String),

    #[error("more than one client found named {0}")]
    TooManyClients(String),

    #[error("can't convert client chunk to JSON")]
    ToJson(#[source] serde_json::Error),

    #[error("failed to generate a chunk key for client {0}")]
    Generate(String, #[source] obnam::client::ClientChunkError),

    #[error("client chunk is not encoded as a data chunk")]
    NotClient,

    #[error("failed to generate a default client key")]
    GenerateKey(#[source] obnam::client::ClientChunkError),

    #[error("failed to create a new credential value")]
    NewCredential(#[source] obnam::credential::CredentialError),
}
